//******************************************************************************
//   MSP-FET430P140 Demo - SMBus Slave Transmits to SMBus Master Receiver
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. The master 
//   reads from the slave. This is the slave code. The master code is called
//   fet140_SMB_mstr.c. The TX data begins at 1 and is incremented 
//   each time it is sent. 
//   Each byte transfer is delayed using TimerA0 to create a timeout condition.
//   The TXRDYIFG interrupt is used to know when to TX.
//   ACLK = n/a, MCLK = SMCLK = I2CIN = DCO ~ 800kHz
//   //* MSP430F169 Device Required *//
//   
//                                 /|\  /|\
//                  MSP430F169     10k  10k     MSP430F169
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.1|<-|----+>|P3.1         P1.0|-->LED
//             |                 |  |      |                 |
//             |                 |  |      |                 |
//             |             P3.3|<-+----->|P3.3             |
//             |                 |         |                 |
//
//
//  H. Grewal
//  Texas Instruments Inc.
//  Feb 2005
//  Built with IAR Embedded Workbench Version: 3.21A
//******************************************************************************

#include  <msp430x16x.h>

typedef enum {
             SMBS_MISC               = (unsigned char) 0x04,
             SMBS_DEVICE_DFLT        = (unsigned char) 0x61,
             SMBS_ALERT_RESPONSE     = (unsigned char) 0x0C,
             SMBS_HOST               = (unsigned char) 0x08,
             SMBS_10BIT_ADD          = (unsigned char) 0x78,
             SMBS_DFLT_ADD           = (unsigned char) 0x37,
             SMBS_ACCESS_HOST        = (unsigned char) 0x28,             
             SMBS_CBUS               = (unsigned char) 0x01,
             SMBS_GCAL               = (unsigned char) 0x00
             }SMBS_Address_t;               
/*---------------------------------------------------------------------------*/	 
                 

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add);           
            
char TXData = 1;

void main (void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT
  P3SEL |= 0x0A;                            // Select I2C pins
  P1DIR |= 0xFF;                            // P1.0 output
                                            // Unused ports to output
  P2DIR =  0xFF;
  P3DIR =  0xFF;
  P4DIR =  0xFF;
  P5DIR =  0xFF;
  P6DIR =  0xFF;
  P1OUT =  0;
  TACTL = TASSEL_2;                         // SMCLK, upmode
  TACCR0 = 22000;
  SMBS_Init (SMBS_MISC,0x48);               // Own address configured as 48h
 
   while(1)
   {
     TACCTL0 = CCIE;
     TACTL |= MC_1;                         // Up Mode
     _BIS_SR(CPUOFF+GIE);                   // Enter LPM0, Enable interrupts
     TACCTL0 &= ~CCIE;
     TACTL &= ~MC_1;
     I2CIE = TXRDYIE;     
   }
}

void SMBS_Init (SMBS_Address_t Add_Param,unsigned char Slave_Add)
{
  U0CTL |= I2C + SYNC;                      // Recommended init procedure
  U0CTL &= ~I2CEN;                          // Recommended init procedure
  I2CTCTL |= I2CSSEL1;                      // SMCLK
  if (Add_Param == SMBS_MISC)
  {
      I2COA = Slave_Add;                    // Own Address is 048h
  } 
  else
      I2COA = (unsigned char) Add_Param ;
  
  I2CIE = TXRDYIE;                          // Enable TXRDYIFG interrupt
  U0CTL |= I2CEN;                           // Enable I2C
}


// Common ISR for I2C Module
#pragma vector=USART0TX_VECTOR
__interrupt void I2C_ISR(void)
{
 
 switch( I2CIV )
 {
   volatile unsigned int i;
   case  2: break;                          // Arbitration lost
   case  4: break;                          // No Acknowledge
   case  6: break;                          // Own Address
   case  8: break;                          // Register Access Ready
   case 10: break;                          // Receive Ready
   case 12:                                 // Transmit Ready
   I2CIE &= ~TXRDYIE;
   I2CDRB = TXData++;                       // Load I2CDRB and increment
            break;
   case 14: break;                          // General Call
   case 16: break;                          // Start Condition
 }
}

#pragma vector=TIMERA0_VECTOR
__interrupt void ta0_isr(void)
{
  _BIC_SR_IRQ(CPUOFF);                      // Clear LPM0
}// ta0_isr()
